/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "joystick.h"
#include "cpc.h"
#include "asic.h"
#include "multiplay.h"
#include <math.h>

typedef struct
{
	/* auto-fire timer; updated */
	int Timer;
	/* auto-fire timer rate */
	int TimerReload;
	/* state of button previously; this is actually a bit mask but with one bit set */
	unsigned long PreviousButtonPressed;
	/* true if auto-fire is active, false otherwise */
	BOOL bActive;
  /* indicate we want it held down for a second frame */
  BOOL bButtonPressed;
 } AutoFireSimulation;

/* this is the analogue data */
typedef struct
{
    /* current x,y */
	signed int x;
	signed int y;
	/* bitmask of buttons that are pressed */
	int buttons;
	/* min/max x range */
	signed int minx, maxx;
	/* min/max y range */
	signed int miny, maxy;
} JoystickData;

typedef struct
{
	/* joystick is active? yes/no */
	BOOL Active;

	int Type;

	/* id of CPC joystick 0,1 are digital, 2,3 are analogue*/
	int CPCJoystickID;

    /* auto fire settings for each button */
	AutoFireSimulation autoFire[CPC_JOYSTICK_NUM_BUTTONS];

    JoystickData Data;

    /* when joystick is real, this is it's id; id is platform specific. it may be an index or a hash value for example */
    int RealJoystickID;

	/* this is the chosen key set; used for ui */
    int nKeySet;

	/* this is initialised with the appropiate key-set. It indicates which physical keys on the host computer
	simulates joysticks on the cpc */
    long SimulatedKeyState;
    int SimulatedKeyIds[JOYSTICK_SIMULATED_KEYID_LAST];

	/* this is initialised with the appropiate mouse state. It indicates which physical mouse inputs on the host
	computer simulates joysticks on the cpc */
    long SimulatedMouseState;
    int SimulatedMouseIds[JOYSTICK_SIMULATED_MOUSEID_LAST];

	/* this array indicates which physical button on a gamepad on the host is mapped to which cpc joystick input. */
	int PhysicalButtonMapping[CPC_JOYSTICK_NUM_BUTTONS];

	/* this array defines which axis is used for X movement and which is for Y movement */
  /* 0 is X, 1 is Y */
	int PhysicalAxisMapping[CPC_JOYSTICK_NUM_AXES];

} JoystickConfigData;

/* NOTE: The joystick state on the keyboard matrix is the result of a key OR joystick */
/* The state will show pressed if a key AND/OR joystick button are pressed, and show not pressed
if neither are pressed */

/* state generated by joystick; bit is 0 if pressed, 1 otherwise */
static  unsigned char KeyboardLine_FromJoysticks[CPC_NUM_DIGITAL_JOYSTICKS];

/* by default don't enable so keyboard doesn't clash with joystick. User needs to turn it on */
BOOL KeyStickActive = FALSE;

void Joystick_KeyboardLine_Reset()
{
  int i;
  for (i = 0; i<CPC_NUM_DIGITAL_JOYSTICKS; i++)
  {
      KeyboardLine_FromJoysticks[i] = 0x0ff;
  }
}
#if 0
void Joystick_KeyboardLine_FromKeyboard(int nLine, int nBit, BOOL bState)
{
    KeyboardLine_FromKeys[nLine]|=(1<<nBit);
    if (bState)
    {
        KeyboardLine_FromKeys[nLine]&=~(1<<nBit);
    }
}
#endif
void Joystick_KeyboardLine_FromJoystick_Reset(int nLine)
{
    KeyboardLine_FromJoysticks[nLine] = 0x0ff;
}

void Joystick_KeyboardLine_FromJoystick(int nLine, int nBit, BOOL bState)
{
    KeyboardLine_FromJoysticks[nLine]|=(1<<nBit);
    if (bState)
    {
        KeyboardLine_FromJoysticks[nLine]&=~(1<<nBit);
    }
}
#if 0
void Joystick_KeyboardLine_Refresh()
{
  int i;
  for (i=0; i<CPC_NUM_DIGITAL_JOYSTICKS; i++)
  {
      int k;
    unsigned char Combined = KeyboardLine_FromKeys[i]&KeyboardLine_FromJoysticks[i];
    
      for (k=0; k<7; k++)
    {
        
        if (Combined & (1<<k))
        {
            /* not pressed by either keyboard or joystick */
            CPC_ClearKeyInternal(JoyKeys[i][k]);
        }
        else
        {
            CPC_SetKeyInternal(JoyKeys[i][k]);
        }
      }
    }
  }
#endif
  
void Joystick_KeyStickActive(BOOL state)
{
	KeyStickActive = state;
}

BOOL Joystick_IsKeyStickActive()
{
	return KeyStickActive;
}

unsigned char Joystick_AdjustValueBasedOnHardware(int nJoystick, unsigned char Data)
{
	/* single joystick on aleste */
	if ((CPC_GetHardware() == CPC_HW_ALESTE) && (nJoystick == 1))
	{
		return 0x0ff;
	}

	/* force bit 7 */
	Data |= (1 << 7);

	/* force spare */
	if ((CPC_GetHardware() == CPC_HW_CPCPLUS) || (CPC_GetHardware() == CPC_HW_ALESTE))
	{
		Data |= (1 << 6);
	}

	return Data;
}

unsigned char Joystick_KeyboardLine_Refresh2(int nJoystick, unsigned char KeyboardJoy)
{	
	unsigned char Data = KeyboardLine_FromJoysticks[nJoystick];
	Data = Joystick_AdjustValueBasedOnHardware(nJoystick, Data);
	return KeyboardJoy & Data;
}

unsigned char Joystick_KeyboardLine_Refresh3(int nJoystick, unsigned char KeyboardJoy, unsigned char JoystickData)
{
	unsigned char Data = JoystickData;
	Data = Joystick_AdjustValueBasedOnHardware(nJoystick, Data);
	return KeyboardJoy & Data;
}


/* configurations for the joysticks */
/* 2 digital, and 2 analogue */
static JoystickConfigData joystickConfigs[CPC_NUM_JOYSTICKS];

/* get the mapping; the return is the physical joystick button. */
int Joystick_GetButtonMapping(int nID,int nButton)
{
  int i;
  
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS)) return -1;

  for (i=0; i<CPC_JOYSTICK_NUM_BUTTONS; i++)
  {
        if (joystickConfigs[nID].PhysicalButtonMapping[i]==nButton)
        {
            return i;
        }
  }
  return -1;
}



/* set the mapping */
void Joystick_SetButtonMapping(int nID,int nButton, int nCPCButton)
{
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS)) return;
    if ((nCPCButton<0) || (nCPCButton>=CPC_JOYSTICK_NUM_BUTTONS)) return;
      
	joystickConfigs[nID].PhysicalButtonMapping[nCPCButton] = nButton;
}


int Joystick_GetButtonMappingCPC(int nID,int nCPCButton)
{
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS)) return -1;
    if ((nCPCButton<0) || (nCPCButton>=CPC_JOYSTICK_NUM_BUTTONS)) return -1;
      
	return joystickConfigs[nID].PhysicalButtonMapping[nCPCButton];
}


/* set the mapping */
void Joystick_SetButtonMappingCPC(int nID,int nCPCButton, int nButton)
{
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS)) return;
    if ((nCPCButton<0) || (nCPCButton>=CPC_JOYSTICK_NUM_BUTTONS)) return;
      
	joystickConfigs[nID].PhysicalButtonMapping[nCPCButton] = nButton;
}

/* get the mapping; the return is the physical joystick Axis. */
int Joystick_GetAxisMappingPhysical(int nID,int nAxis)
{
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS)) return -1;
	if ((nAxis<0) || (nAxis >= CPC_JOYSTICK_NUM_AXES)) return -1;
	
	return joystickConfigs[nID].PhysicalAxisMapping[nAxis];
}

/* Set the mapping for the axis */
void Joystick_SetAxisMappingPhysical(int nID,int nAxis, int nValue)
{
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS)) return;
	if ((nAxis<0) || (nAxis >= CPC_JOYSTICK_NUM_AXES)) return;
	
	joystickConfigs[nID].PhysicalAxisMapping[nAxis] = nValue;
}

void Joystick_SetKeySet(int nID, int nSet)
{
  if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
  {
    return;
  }
  
   joystickConfigs[nID].nKeySet = nSet;
}

int Joystick_GetKeySet(int nID)
{
   if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
   {
    return JOYSTICK_KEYSET_UNKNOWN;
   }
  
   return joystickConfigs[nID].nKeySet;
}

void Joystick_SetSimulatedKeyID(int nID, int nKeyID, int nKeyCode)
{
     if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
     {
      return;
     }
    joystickConfigs[nID].SimulatedKeyIds[nKeyID] = nKeyCode;
}

int Joystick_GetSimulatedKeyID(int nID, int nKeyID)
{
       if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return CPC_KEY_NULL;
       }
       
    return joystickConfigs[nID].SimulatedKeyIds[nKeyID];

}

void Joystick_SetSimulatedKeyIDState(int nID, int nKeyID, BOOL bState)
{
      if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return;
       }
	  
    if (bState)
    {
        joystickConfigs[nID].SimulatedKeyState |= (1<<nKeyID);
    }
	else
	{
		joystickConfigs[nID].SimulatedKeyState &= ~(1 << nKeyID);
	}
}


void Joystick_SetSimulatedMouseID(int nID, int nKeyID, int nKeyCode)
{
        if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return;
       }

    joystickConfigs[nID].SimulatedMouseIds[nKeyID] = nKeyCode;
}

int Joystick_GetSimulatedMouseID(int nID, int nKeyID)
{
  /* todo: extra checks */
    return joystickConfigs[nID].SimulatedMouseIds[nKeyID];

}

void Joystick_SetSimulatedMouseIDState(int nID, int nKeyID, BOOL bState)
{
       if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return;
       }

  
    joystickConfigs[nID].SimulatedMouseState &= ~(1<<nKeyID);
    if (bState)
    {
        joystickConfigs[nID].SimulatedMouseState |= (1<<nKeyID);
    }
}

/* match a host specific joystick id to a cpc joystick id */
int Joystick_PhysicalToCPC(int nPhysical)
{
    int i;
    for (i=0; i<CPC_NUM_JOYSTICKS; i++)
    {
        if (joystickConfigs[i].Type == JOYSTICK_TYPE_REAL)
        {
            if (joystickConfigs[i].RealJoystickID == nPhysical)
            {
                return joystickConfigs[i].CPCJoystickID;
            }
        }
    }

    return -1;
}


/*Return ID in joystick already loaded list. */
int Joystick_PhysicalToID(int nPhysical)
{
   if (joystickConfigs[nPhysical].Type == JOYSTICK_TYPE_REAL)
   {
	return joystickConfigs[nPhysical].CPCJoystickID;
    }

    return -1;
}
void Joystick_SetButton(int nID, int nButton, BOOL bState)
{
       if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return;
       }
  
    joystickConfigs[nID].Data.buttons &= ~(1<<nButton);
    if (bState)
    {
        joystickConfigs[nID].Data.buttons |= (1<<nButton);
    }
}

int Joystick_GetPhysical(int nID)
{
       if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return -1;
       }
  
    return joystickConfigs[nID].RealJoystickID;
}

void Joystick_SetPhysical(int nID, int nPhysical)
{
         if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return;
       }

  
    joystickConfigs[nID].RealJoystickID = nPhysical;
}


void Joystick_SetType(int nID, int nType)
{
          if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
       {
         return;
       }
  
    joystickConfigs[nID].Type = nType;
}

int Joystick_GetType(int nID)
{
	if ((nID<0) || (nID>=CPC_NUM_JOYSTICKS))
    {
		return JOYSTICK_TYPE_UNKNOWN;
    }
  
    return joystickConfigs[nID].Type;
}
#if 0
void Joystick_SetXDigitalMovement(int nID, int nMovement)
{
   JoystickData *pData = &joystickConfigs[nID].Data;
    pData->minx = 0;
    pData->maxx = 32767;

    if (nMovement<0)
    {
        pData->x = pData->minx;
    }
    else if (nMovement>0)
    {
        pData->x = pData->maxx;
    }
    else
    {
        pData->x = ((pData->maxx-pData->minx)>>1)+pData->minx;
    }
}


void Joystick_SetYDigitalMovement(int nID, int nMovement)
{
    JoystickData *pData = &joystickConfigs[nID].Data;
    pData->miny = 0;
    pData->maxy = 32767;

    if (nMovement<0)
    {
        pData->y = pData->miny;
    }
    else if (nMovement>0)
    {
        pData->y = pData->maxy;
    }
    else
    {
        pData->y = ((pData->maxy-pData->miny)>>1)+pData->miny;
    }
}
#endif
void Joystick_SetXRange(int nID, signed int minx, signed int maxx)
{
   JoystickData *pData = &joystickConfigs[nID].Data;
   pData->maxx = maxx;
   pData->minx = minx;
}

void Joystick_SetYRange(int nID, signed int miny, signed int maxy)
{
   JoystickData *pData = &joystickConfigs[nID].Data;
   pData->maxy = maxy;
   pData->miny = miny;

}

void Joystick_SetXMovement(int nID, signed int curx)
{
  JoystickData *pData = &joystickConfigs[nID].Data;
  pData->x = curx;
}

void Joystick_SetYMovement(int nID, signed int cury)
{
  JoystickData *pData = &joystickConfigs[nID].Data;
  pData->y = cury;
}

void Joystick_InitAutoFire(AutoFireSimulation *pAutoFire, BOOL bActive, int Rate)
{
    pAutoFire->bActive = bActive;
    pAutoFire->TimerReload = Rate;
    pAutoFire->Timer = 0;
    pAutoFire->PreviousButtonPressed = 0;
  pAutoFire->bButtonPressed = FALSE;
}

/* ButtonPressed will be 0 or (1<<Button) */
/* return value is 0 or (1<<Button) */
/* fire button is pressed first time button is pressed and then repeats at
the rate defined until button is released */
unsigned long Joystick_UpdateAutoFire(AutoFireSimulation *pAutoFire, unsigned long ButtonPressed)
{
    unsigned long Output;

    /* is it active? */
    if (pAutoFire->bActive)
    {
        
        /* mask off button press; if button is not pressed we show this */
        Output = 0;

        /* button pressed */
        if (ButtonPressed)
        {
          
          
          /* button is pressed and held? because it was pressed this time and pressed last time */
            if (pAutoFire->PreviousButtonPressed)
            {
              /* if button was pressed, hold it down for 2 frames before releasing */
                if (pAutoFire->bButtonPressed)
                {
                    Output = ButtonPressed;
                    pAutoFire->bButtonPressed = FALSE;
                  
                }
              
                /* previous button was pressed */
                /* decrement timer */
                  if (pAutoFire->Timer!=0)
                  {
                      pAutoFire->Timer--;
                  }
            }
            else
            {
              /* button has been pressed for first time */
              
                /* previous button not pressed */
                pAutoFire->Timer = 0;
            }

            /* trigger if pressed for first time */
            if (pAutoFire->Timer<=0)
            {
                /* trigger autofire */
                /* reload timer */
                pAutoFire->Timer = pAutoFire->TimerReload;
              
              /* indicate we want to hold it down for another frame */
                pAutoFire->bButtonPressed = TRUE;
              
                /* allow button press */
                Output = ButtonPressed;
            }
        }
    }
    else
    {
        /* allow button state */
        Output = ButtonPressed;
    }

    pAutoFire->PreviousButtonPressed = ButtonPressed;

    return Output;
}


void	Joystick_InitDefaultSettings(void)
{
    int i;
  
  
  

    for (i=0; i<CPC_NUM_JOYSTICKS; i++)
    {
        int b;
        if (i==0)
        {
            joystickConfigs[i].Active = TRUE;
            joystickConfigs[i].nKeySet = JOYSTICK_KEYSET_NUMPAD;
			joystickConfigs[i].Type = JOYSTICK_TYPE_SIMULATED_BY_KEYBOARD;
		}
        else
        {

            joystickConfigs[i].Active = FALSE;
            joystickConfigs[i].nKeySet = JOYSTICK_KEYSET_UNKNOWN;
			joystickConfigs[i].Type = JOYSTICK_TYPE_UNKNOWN;
        }
        joystickConfigs[i].Data.minx = -32768;
        joystickConfigs[i].Data.maxx = 32767;
		joystickConfigs[i].Data.miny = -32768;
		joystickConfigs[i].Data.maxy = 32767;
		joystickConfigs[i].RealJoystickID = -1;
      /* clear mappings */
        for (b=0; b<CPC_JOYSTICK_NUM_BUTTONS; b++)
        { 
            joystickConfigs[i].PhysicalButtonMapping[b] = -1;
        }
		for (b = 0; b<CPC_JOYSTICK_NUM_AXES; b++)
        { 
            joystickConfigs[i].PhysicalAxisMapping[b] = -1;
        }
      

        Joystick_Reset(i);

        for (b=0; b<CPC_JOYSTICK_NUM_BUTTONS; b++)
        {
            Joystick_InitAutoFire(&joystickConfigs[i].autoFire[b], FALSE, MAXAUTOFIRERATE);
        }
        
        /* the following gives us a default setup where 1 button is mapped to each of cpc's joystick buttons */
        /* the hat is also mapped to the axes */
        /* the direction is also mapped to the axes */
        
          /* map cpc joystick button 0,1,2 to joypad button 0,1,2 */
      Joystick_SetButtonMapping(i,0,0);
      Joystick_SetButtonMapping(i,1,1);
      Joystick_SetButtonMapping(i,2,2);
          /* map axis 0 to pad axis 0 */
          /* map axis 1 to pad axis 1 */
      Joystick_SetAxisMappingPhysical(i,0,0);
      Joystick_SetAxisMappingPhysical(i,1,1);
       
        }
	for (i = 0; i < CPC_NUM_JOYSTICKS; i++)
	{
		joystickConfigs[i].CPCJoystickID = (CPC_JOYSTICK_ID)i;
	}
}


void Joystick_Activate(int nID, BOOL bState)
{
    joystickConfigs[nID].Active = bState;
}

void JoystickAF_Activate(int nID, BOOL bState)
{
	int i;
  /* activate for all 4 joystick buttons */
	for (i=0; i<CPC_JOYSTICK_NUM_BUTTONS; i++)
	{
		joystickConfigs[nID].autoFire[i].bActive = bState;
	}
}
int JoystickAF_GetRate(int nID)
{
	return joystickConfigs[nID].autoFire[0].TimerReload;
}
void JoystickAF_SetRate(int nID, int rate)
{
	int i;
  if (rate<MINAUTOFIRERATE)
  {
    rate = MINAUTOFIRERATE;
  }
  else if (rate>MAXAUTOFIRERATE)
  {
    rate = MAXAUTOFIRERATE;
  }
  
	for (i=0; i<CPC_JOYSTICK_NUM_BUTTONS; i++)
	{
		joystickConfigs[nID].autoFire[i].TimerReload = rate;
   }
}

void Joystick_ResetAll()
{
	// if mouse is controlling joystick, stop movement when leaving window
	for (int i = 0; i < CPC_NUM_JOYSTICKS; i++)
	{
		Joystick_Reset(i);
	}
}

void Joystick_Reset(int nID)
{
	int b;
	for (b=0; b<CPC_JOYSTICK_NUM_BUTTONS; b++)
	{

		joystickConfigs[nID].autoFire[b].bButtonPressed = 0;
		joystickConfigs[nID].autoFire[b].PreviousButtonPressed= 0;
		joystickConfigs[nID].autoFire[b].Timer = joystickConfigs[nID].autoFire[b].TimerReload;
	}

    joystickConfigs[nID].Data.x =((joystickConfigs[nID].Data.maxx-joystickConfigs[nID].Data.minx)>>1)+joystickConfigs[nID].Data.minx;
    joystickConfigs[nID].Data.y =((joystickConfigs[nID].Data.maxy-joystickConfigs[nID].Data.miny)>>1)+joystickConfigs[nID].Data.miny;

   joystickConfigs[nID].SimulatedKeyState = 0;
   joystickConfigs[nID].SimulatedMouseState = 0;

}

BOOL Joystick_IsActive(int nID)
{
    return joystickConfigs[nID].Active;
}

BOOL JoystickAF_IsActive(int nID)
{
    return joystickConfigs[nID].autoFire[0].bActive;
}

#define NUM_KEYS 7

BOOL	Joystick_IsKeyUsedByKeyStick(int Key)
{
	int i;
	
	for (i = 0; i < CPC_NUM_JOYSTICKS; i++)
	{

		/* active? */
		if (joystickConfigs[i].Active)
		{
			if (joystickConfigs[i].Type == JOYSTICK_TYPE_SIMULATED_BY_KEYBOARD)
			{
				int j;

				for (j = 0; j < JOYSTICK_SIMULATED_KEYID_LAST; j++)
				{
					if (joystickConfigs[i].SimulatedKeyIds[j] == Key)
						return TRUE;
				}
			}
		}
	}
	return FALSE;
}

#define DIRECTION_EAST (1<<0)
#define DIRECTION_NORTH (1<<1)
#define DIRECTION_WEST (1<<2)
#define DIRECTION_SOUTH (1<<3)

static int DirectionBitMask[9] =
{
	0,
	DIRECTION_EAST,
	DIRECTION_EAST | DIRECTION_NORTH,
	DIRECTION_NORTH,
	DIRECTION_NORTH | DIRECTION_WEST,
	DIRECTION_WEST,
	DIRECTION_WEST | DIRECTION_SOUTH,
	DIRECTION_SOUTH,
	DIRECTION_SOUTH | DIRECTION_EAST
};

#define DIRECTION_PI 3.141592654f
#define DIRECTION_TWO_PI (DIRECTION_PI*2.0f)
/* full circle split into 8 regions */
#define DIRECTION_OCTANT (DIRECTION_TWO_PI/8.0f)
/* we allow +/- half the region on either side */
#define DIRECTION_RANGE (DIRECTION_OCTANT/2.0f)

int Joystick_Digital_CheckDeadZone(JoystickData *pJoystickData)
{
	/* find mid point which may be offset from 0 */
	signed int midx = ((pJoystickData->maxx - pJoystickData->minx) >> 1) + pJoystickData->minx;
	signed int midy = ((pJoystickData->maxy - pJoystickData->miny) >> 1) + pJoystickData->miny;
	/* calculate x and y distance from mid point */
	signed int dx = pJoystickData->x - midx;
	signed int dy = pJoystickData->y - midy;

	/* work out range (max of x and y in all directions) */
	signed int xminrange = (midx - pJoystickData->minx);
	signed int xmaxrange = (pJoystickData->maxx - midx);
	signed int yminrange = (midy - pJoystickData->miny);
	signed int ymaxrange = (pJoystickData->maxy - midy);
	signed int xrange = (xminrange > xmaxrange) ? xminrange : xmaxrange;
	signed int yrange = (yminrange > ymaxrange) ? yminrange : ymaxrange;
	signed int range = (xrange > yrange) ? xrange : yrange;
	signed int deadzone = ((range * 20) / 100);

	unsigned int dxsq = (dx*dx);
	unsigned int dysq = (dy*dy);
	unsigned int deadsq = (deadzone*deadzone);

	/* if it's in dead zone ignore */
	unsigned int hyp = dxsq + dysq;
	if (hyp <= deadsq)
	{
		return 0;
	}

	{
		int direction;
		/* 0 to 180 in anti-clockwise direction */
		/* 0 to -180 in clockwise direction */
		float angle = atan2f((float)-dy, (float)dx);
		if (angle < 0.0f)
		{
			angle = DIRECTION_TWO_PI + angle;
		}
		angle = angle + DIRECTION_RANGE;
		if (angle > DIRECTION_TWO_PI)
		{
			angle = angle - DIRECTION_TWO_PI;
		}
		direction = (int)floorf(angle/ DIRECTION_OCTANT);
		direction++;

		if (direction > (sizeof(DirectionBitMask) / sizeof(DirectionBitMask[0])))
			return 0;

		return DirectionBitMask[direction];
	}
}

void	Joystick_Update(void)
{
	int i;

  /* reset joystick states from joysticks */
	for (i = 0; i < CPC_NUM_DIGITAL_JOYSTICKS; i++)
	{
		Joystick_KeyboardLine_FromJoystick_Reset(i);
	}
  
	for (i=0; i<CPC_NUM_JOYSTICKS; i++)
	{

		/* active? */
		if (joystickConfigs[i].Active)
		{
		    JoystickConfigData *pConfig = &joystickConfigs[i];
			JoystickData *pJoystickData = &pConfig->Data;
			BOOL bTypeRecognised = FALSE;
     
            /* if simulated, generate data */
            switch (pConfig->Type)
          {
            default:
              break;

            /* if type is unknown, continue */
              case JOYSTICK_TYPE_REAL:
                  bTypeRecognised = TRUE;
                  break;
      
			  case JOYSTICK_TYPE_SIMULATED_BY_TOUCH:
				  bTypeRecognised = TRUE;
				  break;

			  case JOYSTICK_TYPE_SIMULATED_BY_KEYBOARD:
			  {
				  bTypeRecognised = TRUE;
				  pJoystickData->buttons = 0;

				  /* if key stick is not active, ignore keyboard simulating joystick */
				  if (!KeyStickActive)
				  {
					  pJoystickData->x = ((pJoystickData->maxx - pJoystickData->minx) >> 1) + pJoystickData->minx;
					  pJoystickData->y = ((pJoystickData->maxy - pJoystickData->miny) >> 1) + pJoystickData->miny;
					  break;
				  }

                if (
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_UP_KEYID)) ||
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_LEFT_UP_KEYID)) ||
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_RIGHT_UP_KEYID))
                    )
                {
                    pJoystickData->y = pJoystickData->miny;
                }
                else
                if
                    (
                     (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_DOWN_KEYID)) ||
                        (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_LEFT_DOWN_KEYID)) ||
                        (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_RIGHT_DOWN_KEYID))
                        )
                {
                    pJoystickData->y = pJoystickData->maxy;
                }
                else
                {
                    pJoystickData->y = ((pJoystickData->maxy-pJoystickData->miny)>>1)+pJoystickData->miny;
                }

                if (
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_LEFT_KEYID)) ||
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_LEFT_UP_KEYID)) ||
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_LEFT_DOWN_KEYID))
                    )
                {
                    pJoystickData->x = pJoystickData->minx;
                }
                else
                if (
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_RIGHT_KEYID)) ||
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_RIGHT_UP_KEYID)) ||
                    (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_RIGHT_DOWN_KEYID))
                    )
                {
                    pJoystickData->x = pJoystickData->maxx;
                }
                else
                {
                    pJoystickData->x = ((pJoystickData->maxx-pJoystickData->minx)>>1)+pJoystickData->minx;
                }

                if (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_FIRE1_KEYID))
                {
                    pJoystickData->buttons |= (1<<0);
                }
                if (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_FIRE2_KEYID))
                {
                    pJoystickData->buttons |= (1<<1);
                }
                if (pConfig->SimulatedKeyState & (1<<JOYSTICK_SIMULATED_FIRE3_KEYID))
                {
                    pJoystickData->buttons |= (1<<2);
                }
            }
            break;


            /* if simulated, generate data */
            case JOYSTICK_TYPE_SIMULATED_BY_MOUSE:
            {
              bTypeRecognised = TRUE;
                pJoystickData->buttons = 0;

                if (pConfig->SimulatedMouseState  & (1<<JOYSTICK_SIMULATED_FIRE1_MOUSEID))
                {
                    pJoystickData->buttons |= (1<<0);
                }
                if (pConfig->SimulatedMouseState  & (1<<JOYSTICK_SIMULATED_FIRE2_MOUSEID))
                {
                    pJoystickData->buttons |= (1<<1);
                }
                if (pConfig->SimulatedMouseState  & (1<<JOYSTICK_SIMULATED_FIRE3_MOUSEID))
                {
 
                    pJoystickData->buttons |= (1<<2);
                }
            }
            break;
          }

          if (bTypeRecognised)
          {
			/* joyData is setup with information. It contains analogue information
			which has been processed and remapped from original control system (i.e. real or simulated) */
			switch (pConfig->CPCJoystickID)
			{
				case CPC_DIGITAL_JOYSTICK0:
				case CPC_DIGITAL_JOYSTICK1:
				{
					int direction = Joystick_Digital_CheckDeadZone(pJoystickData);

					if ((direction & DIRECTION_WEST)!=0)
					{
            /* left */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 2, TRUE);
           }
					else if ((direction & DIRECTION_EAST)!=0)
					{
            /* right */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 3, TRUE);
					}

					if ((direction & DIRECTION_NORTH)!=0)
					{
            /* up */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 0, TRUE);
					}
					else if ((direction & DIRECTION_SOUTH)!=0)
					{
            /* down */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 1, TRUE);
					}

                    if (Joystick_UpdateAutoFire(&pConfig->autoFire[0], (pJoystickData->buttons & (1<<0))))
                    {
                       /* fire 1 */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 4, TRUE);
					}

                    if (Joystick_UpdateAutoFire(&pConfig->autoFire[1], (pJoystickData->buttons & (1<<1))))
                    {
                      /* fire 2 */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 5, TRUE);
					}

  					/* SPARE not connected on PLUS or Aleste */
                    if ((CPC_GetHardware()==CPC_HW_CPC) || (CPC_GetHardware()==CPC_HW_KCCOMPACT))
                    {
                        if (Joystick_UpdateAutoFire(&pConfig->autoFire[2], (pJoystickData->buttons & (1<<2))))
                        {
  
                          /* fire 3 */
                  						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_DIGITAL_JOYSTICK0, 6, TRUE);

                        }
                    }
				}
				break;

				case MULTIPLAY_JOYSTICK0:
				case MULTIPLAY_JOYSTICK1:
				{
					int m;

					if (!Multiplay_IsUsingJoystick(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0))
					{
						continue;
					}

					{
						int direction = Joystick_Digital_CheckDeadZone(pJoystickData);

						for (m = 0; m < MULTIPLAY_NUM_INPUTS; m++)
						{
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, (MULTIPLAY_INPUT)m, FALSE);
						}

						if ((direction & DIRECTION_WEST)!=0)
						{
							/* left */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_LEFT, TRUE);
						}
						else if ((direction & DIRECTION_EAST)!=0)
						{
							/* right */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_RIGHT, TRUE);
						}

						if ((direction & DIRECTION_NORTH)!=0)
						{
							/* up */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_UP, TRUE);
						}
						else if ((direction & DIRECTION_SOUTH)!=0)
						{
							/* down */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_DOWN, TRUE);
						}

						if (Joystick_UpdateAutoFire(&pConfig->autoFire[0], (pJoystickData->buttons & (1 << 0))))
						{
							/* fire 1 */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_FIRE1, TRUE);
						}

						if (Joystick_UpdateAutoFire(&pConfig->autoFire[1], (pJoystickData->buttons & (1 << 1))))
						{
							/* fire 2 */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_FIRE2, TRUE);
						}

						if (Joystick_UpdateAutoFire(&pConfig->autoFire[2], (pJoystickData->buttons & (1 << 2))))
						{
							/* fire 3 */
							Multiplay_SetInput(pConfig->CPCJoystickID - MULTIPLAY_JOYSTICK0, MULTIPLAY_FIRE3, TRUE);
						}
					}
				}
				break;

				/* check, fire1 for joystick 0 and fire 2 for joystick2? */
				case CPC_ANALOGUE_JOYSTICK0:
                case CPC_ANALOGUE_JOYSTICK1:
				{
				    int AnalogueInputBase;

                    if (pConfig->CPCJoystickID==CPC_ANALOGUE_JOYSTICK0)
                    {
                        AnalogueInputBase = 0;
                    }
                    else
                    {
                        AnalogueInputBase = 2;
                    }

					ASIC_SetAnalogueInput( AnalogueInputBase + 0, (unsigned char) floorf( ( ( pJoystickData->x - pJoystickData->minx ) / (float) ( pJoystickData->maxx - pJoystickData->minx ) )*63.0f));
					ASIC_SetAnalogueInput( AnalogueInputBase + 1, (unsigned char) floorf( ( ( pJoystickData->y - pJoystickData->miny ) / (float) ( pJoystickData->maxy - pJoystickData->miny ) )*63.0f ) );

                    if (Joystick_UpdateAutoFire(&pConfig->autoFire[0], (pJoystickData->buttons & (1<<0))))
                    {
                      /* fire 1 */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_ANALOGUE_JOYSTICK0, 4, TRUE);
					}
                    if (Joystick_UpdateAutoFire(&pConfig->autoFire[1], (pJoystickData->buttons & (1<<1))))
                    {
                       /* fire 2 */
						Joystick_KeyboardLine_FromJoystick(pConfig->CPCJoystickID-CPC_ANALOGUE_JOYSTICK0, 5, TRUE);
					}
					/* SPARE not connected on PLUS */
				}
				break;
			}
      
    }
		}

	}
  /* at this point we have refreshed the inputs comming from joystick */
}


